

#include <time.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>
#include"rfc_amalgamated.h"
#include"lice/lice/lice.h"
#include"bass/bass.h"
#include"bass/BASS_WADSP.h"
#include"bass/tags.h"

#pragma comment(lib,"bass/bass.lib")
#pragma comment(lib,"bass/BASS_WADSP.lib")
#pragma comment(lib,"bass/tags.lib")

#define IMG_WIDTH 340
#define IMG_HEIGHT 285

// animation data
#define MAX_BALL_RADIUS 7
#define SPEED_RANGE 2
#define ALPHA_DECAY 80
#define ALPHA_INC 0.08
#define PARTICLE_COUNT  70
//#define BLUR_AMOUNT 9

#define LINES_COUNT 6

#define FRAME_TIME 33

#define COLOR_THEMES_COUNT 8
int colorThemes[COLOR_THEMES_COUNT][3] = { { 117, 86, 250 },
{ 112, 224, 151 },
{ 241, 139, 175 },
{ 238, 245, 141 },
{ 24, 226, 39 },
{ 72, 72, 234 },
{ 255, 201, 14 },
{ 195, 195, 195 }
};

int GetRandomNumber()
{
	return rand() % 255;
}

float GetRandomFloat()
{
	return GetRandomNumber() / (float)255;
}

class Particle
{
public:
	int size, fadeBack;
	float alpha, x, y, xSpeed, ySpeed, alphaDecay;

	Particle()
	{
		x = GetRandomFloat() * IMG_WIDTH;
		y = GetRandomFloat() * IMG_HEIGHT;

		alpha = GetRandomFloat();
		size = GetRandomNumber() % MAX_BALL_RADIUS;

		xSpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
		ySpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
		alphaDecay = GetRandomFloat() / ALPHA_DECAY;

		fadeBack = 0;
	}

	void updateParticle()
	{
		x += xSpeed;
		y += ySpeed;

		alpha -= alphaDecay;

		if (alpha <= 0)
		{
			x = GetRandomFloat() * IMG_WIDTH;
			y = GetRandomFloat() * IMG_HEIGHT;

			xSpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
			ySpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;

			fadeBack = 1;
		}


		if (fadeBack == 1)
		{
			alpha += ALPHA_INC;

			if (alpha >= 1)
			{
				alpha = 1;
				fadeBack = 0;
			}
		}


	}
};



class NOWComponent
{
protected:
	bool compVisible;
	int compX, compY;
	int compWidth, compHeight;


public:
	NOWComponent()
	{
		compX = 0;
		compY = 0;
		compVisible = true;
		compWidth = 0;
		compHeight = 0;
	}

	virtual void SetPosition(int x, int y)
	{
		compX = x;
		compY = y;
	}

	virtual int GetPosX()
	{
		return compX;
	}

	virtual int GetPosY()
	{
		return compY;
	}

	virtual void SetSize(int width, int height)
	{
		compWidth = width;
		compHeight = height;
	}

	virtual int GetWidth()
	{
		return compWidth;
	}

	virtual int GetHeight()
	{
		return compHeight;
	}

	virtual void SetVisible(bool visible)
	{
		compVisible = visible;
	}

	virtual bool IsVisible()
	{
		return compVisible;
	}


	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;

		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			return true;
		}
		return false;
	}

	//events!

	virtual void OnMouseLDown(int xpos, int ypos){}
	
	virtual void OnMouseRDown(int xpos, int ypos){}

	virtual void OnMouseLUp(int xpos, int ypos){}

	virtual void OnMouseRUp(int xpos, int ypos){}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys){}

	virtual void OnMouseOver(WPARAM fwKeys){}

	virtual void OnMouseLost(WPARAM fwKeys){}

	virtual void OnPaint(LICE_SysBitmap *canvas){}

	virtual ~NOWComponent(){}

};

class NOWImage : public NOWComponent
{
protected:
	LICE_IBitmap *image;

public:
	NOWImage()
	{
		image = 0;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_Blit(canvas, image, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void SetImage(char* imagePath)
	{
		image = LICE_LoadPNG(imagePath);

		compWidth = image->getWidth();
		compHeight = image->getHeight();
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;

		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			LICE_pixel p = LICE_GetPixel(image, relX, relY);
			if (LICE_GETA(p) > 100)
				return true;
		}
		return false;
	}

	virtual ~NOWImage()
	{
		if (image)
			delete image;
	}
};

class NOWButtonListener
{
public:
	virtual void OnButtonPress(void* button) = 0;
	virtual void SetMouseFocusTo(NOWComponent *component) = 0;
};

class NOWButton : public NOWComponent
{
protected:
	LICE_IBitmap *normalImage,*downImage;
	bool isOver, isDown,lDown;
	NOWButtonListener *listener;
	bool useAlphaHitTest;
public:
	NOWButton(bool useAlphaHitTest=true)
	{
		normalImage = 0;
		downImage = 0;
		listener = 0;
		this->useAlphaHitTest = useAlphaHitTest;

		isOver = false;
		isDown = false;
		lDown = false;
	}

	virtual void SetImages(char* normalImagePath,char* downImagePath)
	{
		normalImage = LICE_LoadPNG(normalImagePath);
		downImage = LICE_LoadPNG(downImagePath);

		compWidth = normalImage->getWidth();
		compHeight = normalImage->getHeight();
	}

	virtual void SetListener(NOWButtonListener *listener)
	{
		this->listener = listener;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		if (isDown)
			LICE_Blit(canvas, downImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
		else
			LICE_Blit(canvas, normalImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void OnButtonClick()
	{
		if (listener)
			listener->OnButtonPress(this);
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;
		
		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			if (useAlphaHitTest)
			{
				LICE_pixel p = LICE_GetPixel(isDown ? downImage : normalImage, relX, relY);
				if (LICE_GETA(p) > 100)
					return true;
			}
			else
			{
				return true;
			}
		}
		return false;
	}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys)
	{
		if (fwKeys == MK_LBUTTON)
		{
			if (lDown)
			{
				if (IsMouseWithinArea(xpos, ypos))
				{
					isDown = true;
				}
				else
				{
					isDown = false;
				}
			}
		}
	}

	virtual void OnMouseLDown(int xpos, int ypos)
	{
		isDown = true;
		lDown = true;
		if (listener)
			listener->SetMouseFocusTo(this);
	}

	virtual void OnMouseLUp(int xpos, int ypos)
	{
		isDown = false;
		
		if (listener)
			listener->SetMouseFocusTo(0);

		if (lDown)
		{
			if (IsMouseWithinArea(xpos, ypos))
			{
				OnButtonClick();
			}
		}
		lDown = false;
	}

	virtual ~NOWButton()
	{
		if (normalImage)
			delete normalImage;
		if (downImage)
			delete downImage;
	}

};

class NOWSliderListener
{
public:
	virtual void OnSliderChange(void* slider) = 0;
	virtual void SetMouseFocusTo(NOWComponent *component) = 0;
};

class NOWSlider : public NOWComponent
{
protected:
	int value;
	NOWSliderListener *listener;
	bool lDown;
public:
	NOWSlider()
	{
		value = 50;
		compWidth = 150;
		compHeight = 16;
		listener = 0;
		lDown = false;
	}

	virtual void SetListener(NOWSliderListener *listener)
	{
		this->listener = listener;
	}

	virtual void SetValue(int value)
	{
		if (value<0)
		{
			this->value = 0;
		}
		else if (value>100)
		{
			this->value = 100;
		}
		else
		{
			this->value = value;
		}
	}

	virtual int GetValue()
	{
		return value;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_FillRect(canvas, compX, compY + 7, compWidth, 4, LICE_RGBA(0, 0, 0, 255));

		if (value)
			LICE_FillRect(canvas, compX, compY + 7, (int)(compWidth*(value / (float)100)), 4, LICE_RGBA(255, 255, 255, 255));
	}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys)
	{
		if (fwKeys == MK_LBUTTON)
		{
			if (lDown)
			{
				int relX = xpos - compX;

				SetValue((relX / (float)compWidth) * 100);
				if (listener)
					listener->OnSliderChange(this);
			}
		}
	}

	virtual void OnMouseLDown(int xpos, int ypos)
	{
		lDown = true;

		if (listener)
			listener->SetMouseFocusTo(this);

		int relX = xpos - compX;
		SetValue((relX / (float)compWidth) * 100);
		if (listener)
			listener->OnSliderChange(this);
	}

	virtual void OnMouseLUp(int xpos, int ypos)
	{
		lDown = false;

		if (listener)
			listener->SetMouseFocusTo(0);
	}

	virtual ~NOWSlider(){}
};

class NOWGhostSlider : public NOWSlider
{
protected:
	int ghostValue;
public:
	NOWGhostSlider()
	{
		ghostValue = 0;
	}

	virtual void SetGhostValue(int value)
	{
		if (value<0)
		{
			this->ghostValue = 0;
		}
		else if (value>100)
		{
			this->ghostValue = 100;
		}
		else
		{
			this->ghostValue = value;
		}
	}

	virtual void OnMouseLUp(int xpos, int ypos)
	{
		if (lDown)
		{
			SetValue(ghostValue);
			ghostValue = 0;
			if (listener)
				listener->OnSliderChange(this);
		}

		lDown = false;

		if (listener)
			listener->SetMouseFocusTo(0);
	}

	virtual void OnMouseLDown(int xpos, int ypos)
	{
		lDown = true;

		if (listener)
			listener->SetMouseFocusTo(this);

		int relX = xpos - compX;
		SetGhostValue((relX / (float)compWidth) * 100);
	}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys)
	{
		if (fwKeys == MK_LBUTTON)
		{
			if (lDown)
			{
				int relX = xpos - compX;
				SetGhostValue((relX / (float)compWidth) * 100);
			}
		}
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		LICE_FillRect(canvas, compX, compY + 7, compWidth, 4, LICE_RGBA(0, 0, 0, 255));

		if (value)
			LICE_FillRect(canvas, compX, compY + 7, (int)(compWidth*(value / (float)100)), 4, LICE_RGBA(255, 255, 255, 255));

		if (ghostValue)
			LICE_FillRect(canvas, compX, compY + 7, (int)(compWidth*(ghostValue / (float)100)), 4, LICE_RGBA(192, 192, 192, 192), 0.6f);
	}

	virtual ~NOWGhostSlider(){}
};

class NOWToggleButton : public NOWButton
{
protected:
	bool toggled;
public:
	NOWToggleButton() :NOWButton(false)
	{
		toggled = false;
	}

	virtual bool IsToggled()
	{
		return toggled;
	}

	virtual void SetToggleState(bool state)
	{
		toggled = state;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		bool showtoggle = isDown ? !toggled : toggled;

		if (showtoggle)
			LICE_Blit(canvas, downImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
		else
			LICE_Blit(canvas, normalImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual void OnButtonClick()
	{
		toggled = !toggled;

		if (listener)
			listener->OnButtonPress(this);
	}

	virtual ~NOWToggleButton(){}
};

class NOWCheckBox : public NOWToggleButton
{
protected:
	KString text;
	KFont *font;
public:
	NOWCheckBox(KFont *font)
	{
		this->font = font;

		compWidth = 200;
		compHeight = 16;
	}

	RECT CalculateTextSize(const wchar_t *text, HFONT hFont)
	{
		HDC hDC = CreateICW(L"DISPLAY", NULL, NULL, NULL);
		HGDIOBJ hOldFont = SelectObject(hDC, hFont);
		RECT sz = { 0, 0, 0, 0 };
		DrawTextW(hDC, text, lstrlenW(text), &sz, DT_CALCRECT | DT_NOPREFIX);
		SelectObject(hDC, hOldFont);
		DeleteDC(hDC);
		return sz;
	}

	virtual void SetText(KString text)
	{
		this->text = text;

		if (normalImage)
		{
			RECT rect = CalculateTextSize((const wchar_t*)text, font->GetFontHandle());
			compWidth = normalImage->getWidth() + 5 + rect.right;
		}
	}

	virtual void SetImages(char* normalImagePath, char* downImagePath)
	{
		normalImage = LICE_LoadPNG(normalImagePath);
		downImage = LICE_LoadPNG(downImagePath);

		compHeight = normalImage->getHeight();
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		NOWToggleButton::OnPaint(canvas);

		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0, 0, 0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX + normalImage->getWidth()+5;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_LEFT | DT_NOPREFIX | DT_END_ELLIPSIS | DT_SINGLELINE | DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWCheckBox(){}
};

class NOWLabel : public NOWComponent
{
protected:
	KString text;
	KFont *font;
public:
	NOWLabel(KFont *font)
	{
		this->font = font;

		compWidth = 150;
		compHeight = 20;
	}

	virtual void SetText(KString text)
	{
		if (text.GetLength() == 0)
			this->text = L"Unknown";
		else
			this->text = text;
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		return false;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0,0,0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextW(hdc, (const wchar_t*)text, text.GetLength(), &rect, DT_LEFT | DT_NOPREFIX|DT_END_ELLIPSIS|DT_SINGLELINE|DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWLabel(){}
};

class NOWTimeLabel : public NOWComponent
{
protected:
	char text[32];
	KFont *font;
	bool alignLeft;
public:
	NOWTimeLabel(KFont *font,bool alignLeft = true)
	{
		this->alignLeft = alignLeft;

		strcpy(text, "0:00");
		this->font = font;

		compWidth = 45;
		compHeight = 20;
	}

	virtual void SetTime(int hour,int min,int sec)
	{
		if (hour == 0)
		{
			sprintf(text, "%02d:%02d", min, sec);
		}
		else
		{
			sprintf(text, "%d:%02d:%02d", hour, min, sec);
		}
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		return false;
	}

	virtual void OnPaint(LICE_SysBitmap *canvas)
	{
		HDC hdc = canvas->getDC();
		int prevBkMode = SetBkMode(hdc, TRANSPARENT);
		COLORREF prevCol = SetTextColor(hdc, RGB(0, 0, 0));
		HGDIOBJ oldFont = SelectObject(hdc, font->GetFontHandle());

		RECT rect;
		rect.left = compX;
		rect.top = compY;
		rect.right = compX + compWidth;
		rect.bottom = compY + compHeight;
		DrawTextA(hdc, text, strlen(text), &rect, (alignLeft?DT_LEFT:DT_RIGHT)| DT_NOPREFIX | DT_END_ELLIPSIS | DT_SINGLELINE | DT_VCENTER);

		SetTextColor(hdc, prevCol);
		SetBkMode(hdc, prevBkMode);
		SelectObject(hdc, oldFont);
	}

	virtual ~NOWTimeLabel(){}
};

class MainWindow : public KWindow, public KThread, public NOWButtonListener, public NOWSliderListener, public KTimerListener
{
private:
	LICE_SysBitmap *backBuffer, *backPanel;
	LICE_IBitmap *gradImage, *ballImage, *linesImage;
	LICE_IBitmap *shapeImg, *maskImg, *glassImg, *visGridImg;

	Particle *particles[PARTICLE_COUNT];

	int lineCurrentXPos;

	// for fps control
	double pcFreq;

	// client area drag data
	bool windowDraging;
	int cld_mouse_x;
	int cld_mouse_y;

	KPointerList<NOWComponent*> *componentList;
	NOWComponent *mouseOverComponent,*mouseFocusedComponent;

	NOWButton *playBtn, *nextBtn, *prevBtn, *starBtn, *navBtn;
	NOWImage *playbackPanel, *infoPanel, *albumArt, *winampIcon;
	NOWSlider *volSlider;
	NOWGhostSlider *seekBar;
	NOWToggleButton *muteBtn;
	NOWLabel *titleLabel,*artistLabel;
	NOWTimeLabel *elapsedTimeLabel,*totalTimeLabel;
	KFont *normalFont,*headerFont,*smallFont;
	NOWCheckBox *agcCheckBox,*hifiCheckBox;

	int starHideDelay,upperStarPos,upperStarStepIndex;
	int downStarPos;
	bool showFirstHalf;
	LICE_IBitmap *starsImg;

	int colorThemeIndex;
	int currentlyActivePanel;

	HSTREAM hstream;

	KTimer seekBarTimer;

	volatile bool enableVis;
public:

	void DrawWindow()
	{
		// clear back buffer and draw window shape image
		LICE_FillRect(backBuffer, 0, 0, IMG_WIDTH, IMG_HEIGHT, LICE_RGBA(255, 255, 255, 0), 0.0F, LICE_BLIT_MODE_COPY);
		LICE_Blit(backBuffer, shapeImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);
		
		// draw grad img on back panel
		LICE_Blit(backPanel, gradImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_MODE_COPY);

		// draw balls on back panel
		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			particles[i]->updateParticle();
			if (particles[i]->size)
			{
				LICE_Blit(backPanel, ballImage, floor(particles[i]->x), floor(particles[i]->y), 0, 24 + (particles[i]->size * 24), 24, 24, particles[i]->alpha, LICE_BLIT_USE_ALPHA);
			}
		}

		// draw lines on back panel
		LICE_Blit(backPanel, linesImage, 0, (IMG_HEIGHT - 200) / 2, lineCurrentXPos, 0, IMG_WIDTH, 200, 1.0f, LICE_BLIT_USE_ALPHA);
		lineCurrentXPos+=2;
		if (lineCurrentXPos > 943)
			lineCurrentXPos = 0;


		// draw ui components on back panel	
		for (int i = 0; i < componentList->GetSize(); i++)
		{
			NOWComponent* component = componentList->GetPointer(i);
			if (component->IsVisible())
				component->OnPaint(backPanel);
		}

		// draw vis
		#define VIS_POS_X 28
		#define VIS_POS_Y 247
		#define VIS_HEIGHT 24
		#define VIS_WIDTH 43
		LICE_Blit(backPanel, visGridImg, 23, 224, 0, 0, 54, 30, 1.0f, LICE_BLIT_USE_ALPHA | LICE_BLIT_MODE_OVERLAY);
		if (enableVis)
		{
			if (hstream)
			{
				float fft[1024]; // fft data buffer
				BASS_ChannelGetData(hstream, fft, BASS_DATA_FFT256 | BASS_DATA_FFT_REMOVEDC);
				for (int i = 0; i < VIS_WIDTH; i++)
				{
					int val = sqrt(fft[i]) * 3 * VIS_HEIGHT - 4; // scale it (sqrt to make low values more visible)

					if (val<0)
						val *= -1;
					if (val>VIS_HEIGHT)
						val = VIS_HEIGHT;

					LICE_Line(backPanel, VIS_POS_X + i, VIS_POS_Y, VIS_POS_X + i, VIS_POS_Y - val, LICE_RGBA(255, 255, 255, 255), 0.6f);
				}
			}
		}

		// draw stars
		if (starHideDelay>0)
		{
			starHideDelay--;
		}else // show upper & down star
		{
			int starPattern[96] = { -1, 6, 5, 4, 3, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 1, 1, 0, 0, 0, 1, 1, 2, 2, 4, 5, 6, -1,
				-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
				-1, -1, -1, -1, -1, -1, -1,-1, -1, 6, 5, 4, 3, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 1, 1, 0, 0, 0, 1, 1, 2, 2, 4, 5, 6, -1 };

			if ((showFirstHalf & (upperStarStepIndex < 48)) || ((!showFirstHalf)&(48 < upperStarStepIndex)))
			{

				if (starPattern[upperStarStepIndex] != -1) // -1 means do not show
					LICE_Blit(backPanel, starsImg, upperStarPos, 10, 0, 17 + (starPattern[upperStarStepIndex] * 70), 68, 70, 1.0, LICE_BLIT_USE_ALPHA);


				if (starPattern[upperStarStepIndex] != -1) // -1 means do not show
					LICE_Blit(backPanel, starsImg, downStarPos, 170, 0, 17 + (starPattern[upperStarStepIndex] * 70), 68, 70, 1.0, LICE_BLIT_USE_ALPHA);
			}

			upperStarPos += 3;
			upperStarStepIndex++;

			downStarPos -= 3;

			if (upperStarStepIndex == 96) // star came to end
			{
				upperStarPos = -5;
				upperStarStepIndex = 0;

				downStarPos = 280;

				starHideDelay = 100 + (GetRandomNumber() % 100 * 2);
				showFirstHalf = (GetRandomNumber() % 2)==1;
			}
		}

		// draw glass panel on back panel
		LICE_Blit(backPanel, glassImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// draw mask image on back panel and clear pink area
		LICE_Blit(backPanel, maskImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);
		LICE_SetAlphaFromColorMask(backPanel, LICE_RGBA(255, 0, 255, 0));

		// draw back panel on backbuffer
		LICE_Blit(backBuffer, backPanel, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// Create memory DC
		HDC hdcScreen = GetDC(NULL);
		HDC hDC = CreateCompatibleDC(hdcScreen);

		// Create memory bitmap
		HBITMAP hBmp = CreateCompatibleBitmap(hdcScreen, IMG_WIDTH, IMG_HEIGHT);
		HBITMAP hBmpOld = (HBITMAP)SelectObject(hDC, hBmp);

		BitBlt(hDC, 0, 0, compWidth, compHeight, backBuffer->getDC(), 0, 0, SRCCOPY);	

		// Call UpdateLayeredWindow
		BLENDFUNCTION blend = { 0 };
		blend.BlendOp = AC_SRC_OVER;
		blend.SourceConstantAlpha = 255;
		blend.AlphaFormat = AC_SRC_ALPHA;
		SIZE szWnd = { IMG_WIDTH, IMG_HEIGHT };
		POINT ptSrc = { 0, 0 };
		UpdateLayeredWindow(compHWND, hdcScreen, NULL, &szWnd, hDC, &ptSrc, 0, &blend, ULW_ALPHA);

		SelectObject(hDC, hBmpOld);
		DeleteObject(hBmp);
		DeleteDC(hDC);
		ReleaseDC(NULL, hdcScreen);
	}

	void Run() // drawing thread
	{
		
		while (!threadShouldStop)
		{
			LARGE_INTEGER li;
			QueryPerformanceCounter(&li);
			__int64 counterStart = li.QuadPart;

			DrawWindow();

			QueryPerformanceCounter(&li);
			double deltaTime = double(li.QuadPart - counterStart) / pcFreq;

			/*char buff[256];
			sprintf(buff, "%d", (int)deltaTime);
			MessageBoxA(0, buff, buff, 0);*/

			if ((FRAME_TIME - deltaTime)>0)
				Sleep(FRAME_TIME - deltaTime);


		}
		isThreadRunning = false;
	}

	void GenerateGradImage(int themeIndex)
	{
		int h, s, v;
		LICE_RGB2HSV(colorThemes[themeIndex][0], colorThemes[themeIndex][1], colorThemes[themeIndex][2], &h, &s, &v);
		for (int i = 0; i<IMG_HEIGHT; i++)
		{
			int newV = (i / (float)IMG_HEIGHT)*(float)v;
			int r, g, b;
			LICE_HSV2RGB(h, s, newV, &r, &g, &b);
			LICE_Line(gradImage, 0, i, IMG_WIDTH, i, LICE_RGBA(r, g, b, 255), 1.0, LICE_BLIT_MODE_COPY, true);
		}
	}

	void Panel1Visibility(bool visible)
	{
		albumArt->SetVisible(visible);
		winampIcon->SetVisible(visible);
		muteBtn->SetVisible(visible);
		volSlider->SetVisible(visible);
		seekBar->SetVisible(visible);
		titleLabel->SetVisible(visible);
		artistLabel->SetVisible(visible);
		elapsedTimeLabel->SetVisible(visible);
		totalTimeLabel->SetVisible(visible);
		enableVis = visible;
	}

	void Panel2Visibility(bool visible)
	{
		agcCheckBox->SetVisible(visible);
		hifiCheckBox->SetVisible(visible);
	}

	void init()
	{
		LARGE_INTEGER li;
		QueryPerformanceFrequency(&li);
		pcFreq = double(li.QuadPart) / 1000.0;

		starHideDelay = 50 ;
		upperStarPos = -5;
		upperStarStepIndex = 0;
		downStarPos = 280;
		showFirstHalf = 1;

		lineCurrentXPos = 0;

		colorThemeIndex = 0;

		currentlyActivePanel = 0;

		enableVis = true;

		hstream = 0;

		// initialize drag data!
		windowDraging = false;

		componentList = new KPointerList<NOWComponent*>();
		mouseOverComponent = 0;
		mouseFocusedComponent = 0;

		backBuffer = new LICE_SysBitmap(IMG_WIDTH, IMG_HEIGHT);
		backPanel = new LICE_SysBitmap(IMG_WIDTH, IMG_HEIGHT);
		gradImage = new LICE_MemBitmap(IMG_WIDTH, IMG_HEIGHT);
		ballImage = LICE_LoadPNG("c:\\balls.png");
		linesImage = LICE_LoadPNG("c:\\lines.png");

		shapeImg = LICE_LoadPNG("c:\\shape.png");
		maskImg = LICE_LoadPNG("c:\\mask.png");
		glassImg = LICE_LoadPNG("c:\\glasspanel.png");
		starsImg = LICE_LoadPNG("c:\\star.png");
		visGridImg = LICE_LoadPNG("c:\\widgets\\vis-grid.png");

		normalFont = new KFont(L"Tahoma", 15);
		headerFont = new KFont(L"Tahoma", 15, true);
		smallFont = new KFont(L"Tahoma", 12);

		// generate grad img
		GenerateGradImage(colorThemeIndex);

		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			particles[i] = new Particle();
		}

		playbackPanel = new NOWImage();
		playbackPanel->SetImage("c:\\widgets\\playback-panel.png");
		playbackPanel->SetPosition((IMG_WIDTH - 210) / 2, 193);
		AddNowComponent(playbackPanel);

		infoPanel = new NOWImage();
		infoPanel->SetImage("c:\\widgets\\info-panel.png");
		infoPanel->SetPosition((IMG_WIDTH - 328) / 2, 23);
		AddNowComponent(infoPanel);

		albumArt = new NOWImage();
		albumArt->SetImage("c:\\widgets\\album-art.png");
		albumArt->SetPosition(36, 57);
		AddNowComponent(albumArt);

		winampIcon = new NOWImage();
		winampIcon->SetImage("c:\\widgets\\winampa.png");
		winampIcon->SetPosition(125, 87);
		AddNowComponent(winampIcon);

		navBtn = new NOWButton(false);
		navBtn->SetImages("c:\\widgets\\nav-btn-normal.png", "c:\\widgets\\nav-btn-down.png");
		navBtn->SetPosition(296, 95);
		navBtn->SetListener(this);
		AddNowComponent(navBtn);

		playBtn = new NOWButton();
		playBtn->SetImages("c:\\widgets\\play-btn-normal.png", "c:\\widgets\\play-btn-down.png");
		playBtn->SetPosition((IMG_WIDTH - 42) / 2, 218);
		playBtn->SetListener(this);
		AddNowComponent(playBtn);
		
		nextBtn = new NOWButton();
		nextBtn->SetImages("c:\\widgets\\next-btn-normal.png", "c:\\widgets\\next-btn-down.png");
		nextBtn->SetPosition(((IMG_WIDTH - 42) / 2)+46, 218);
		nextBtn->SetListener(this);
		AddNowComponent(nextBtn);

		prevBtn = new NOWButton();
		prevBtn->SetImages("c:\\widgets\\prev-btn-normal.png", "c:\\widgets\\prev-btn-down.png");
		prevBtn->SetPosition(((IMG_WIDTH - 42) / 2) - 46, 218);
		prevBtn->SetListener(this);
		AddNowComponent(prevBtn);

		starBtn = new NOWButton();
		starBtn->SetImages("c:\\widgets\\star-btn-normal.png", "c:\\widgets\\star-btn-down.png");
		starBtn->SetPosition(((IMG_WIDTH - 42) / 2) + 120, 218);
		starBtn->SetListener(this);
		AddNowComponent(starBtn);

		muteBtn = new NOWToggleButton();
		muteBtn->SetImages("c:\\widgets\\mute-btn-normal.png", "c:\\widgets\\mute-btn-toggle.png");
		muteBtn->SetPosition(127, 118);
		muteBtn->SetListener(this);
		AddNowComponent(muteBtn);

		volSlider = new NOWSlider();
		volSlider->SetSize(100, 16);
		volSlider->SetPosition(151, 119);
		volSlider->SetListener(this);
		AddNowComponent(volSlider);

		seekBar = new NOWGhostSlider();
		seekBar->SetSize(254, 16);
		seekBar->SetPosition(43, 176);
		seekBar->SetListener(this);
		AddNowComponent(seekBar);

		titleLabel = new NOWLabel(headerFont);
		titleLabel->SetText(L"Around My Heat (Dj Sveshnikov_Remix)");
		titleLabel->SetSize(169, 20);
		titleLabel->SetPosition(127, 63);
		AddNowComponent(titleLabel);

		artistLabel = new NOWLabel(normalFont);
		artistLabel->SetText(L"Sandra");
		artistLabel->SetSize(148, 20);
		artistLabel->SetPosition(150, 87);
		AddNowComponent(artistLabel);

		elapsedTimeLabel = new NOWTimeLabel(smallFont);
		elapsedTimeLabel->SetTime(0, 1, 5);
		elapsedTimeLabel->SetPosition(45, 160);
		AddNowComponent(elapsedTimeLabel);

		totalTimeLabel = new NOWTimeLabel(smallFont,false);
		totalTimeLabel->SetTime(0, 3, 44);
		totalTimeLabel->SetPosition(251, 160);
		AddNowComponent(totalTimeLabel);

		agcCheckBox = new NOWCheckBox(normalFont);
		agcCheckBox->SetImages("c:\\widgets\\unchecked-box.png", "c:\\widgets\\checked-box.png");
		agcCheckBox->SetText(L"Enable Automatic Gain Control"); // only call this method after setting images!
		agcCheckBox->SetPosition(40, 57);
		agcCheckBox->SetListener(this);
		AddNowComponent(agcCheckBox);

		hifiCheckBox = new NOWCheckBox(normalFont);
		hifiCheckBox->SetImages("c:\\widgets\\unchecked-box.png", "c:\\widgets\\checked-box.png");
		hifiCheckBox->SetText(L"Enable Hi-fi Audio Engine"); // only call this method after setting images!
		hifiCheckBox->SetPosition(40, 80);
		hifiCheckBox->SetListener(this);
		AddNowComponent(hifiCheckBox);

		Panel2Visibility(false);

		SetStyle(WS_POPUP | WS_SYSMENU | WS_MINIMIZEBOX);
		SetExStyle(WS_EX_LAYERED);
		SetSize(IMG_WIDTH, IMG_HEIGHT);
		CreateComponent();
		DrawWindow();
		CenterScreen();
		DragAcceptFiles(compHWND, TRUE);

		BASS_Init(-1, 44100, 0, compHWND, NULL);
		TAGS_SetUTF8(TRUE);

		seekBarTimer.SetTimerWindow(this);
		seekBarTimer.SetInterval(1000);
		seekBarTimer.SetListener(this);
		seekBarTimer.StartTimer();

		this->StartThread(); // start animation drawing
	}

	void OnTimer(KTimer *timer)
	{
		if (timer == &seekBarTimer)
		{
			if (hstream)
			{
				QWORD posByte = BASS_ChannelGetPosition(hstream, BASS_POS_BYTE);
				double totalSecs = BASS_ChannelBytes2Seconds(hstream, posByte);

				int hours, mins, secs;
				ConvertToTime(totalSecs, &hours, &mins, &secs);
				elapsedTimeLabel->SetTime(hours, mins, secs);

				QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);

				seekBar->SetValue((posByte / (float)length) * 100);
			}
		}
	}

	void OnSliderChange(void* slider)
	{
		if (slider == volSlider)
		{
			BASS_SetConfig(BASS_CONFIG_GVOL_STREAM, (volSlider->GetValue()/(float)100)*10000);
		}
		else if (slider == seekBar)
		{
			if (hstream)
			{
				QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);
				BASS_ChannelSetPosition(hstream, (seekBar->GetValue() / (float)100)*length, BASS_POS_BYTE);

				// update elapsed time lable on same time
				QWORD posByte = BASS_ChannelGetPosition(hstream, BASS_POS_BYTE);
				double totalSecs = BASS_ChannelBytes2Seconds(hstream, posByte);

				int hours, mins, secs;
				ConvertToTime(totalSecs, &hours, &mins, &secs);
				elapsedTimeLabel->SetTime(hours, mins, secs);
			}
		}
	}

	void SetMouseFocusTo(NOWComponent *component)
	{
		mouseFocusedComponent = component;
	}

	void OnButtonPress(void* button)
	{
		if (button == starBtn)
		{
			colorThemeIndex++;
			if (colorThemeIndex == COLOR_THEMES_COUNT)
				colorThemeIndex = 0;

			// generate grad img
			GenerateGradImage(colorThemeIndex);
		}
		else if (button == navBtn)
		{
			if (currentlyActivePanel == 0)
			{
				Panel1Visibility(false);
				Panel2Visibility(true);

				currentlyActivePanel++;
			}
			else if (currentlyActivePanel == 1)
			{
				Panel2Visibility(false);
				Panel1Visibility(true);
				
				currentlyActivePanel = 0;
			}
			
		}
	}

	void OnClose()
	{
		seekBarTimer.StopTimer();

		this->ThreadShouldStop();

		while (isThreadRunning){} //wait till animation drawing thread finish

		BASS_Stop();
		if (hstream)
		{
			BASS_StreamFree(hstream);
		}
		BASS_Free();

		DragAcceptFiles(compHWND, FALSE);
		::DestroyWindow(compHWND);
	}

	void OnPaint() // handle random dummy paint msgs
	{
		PAINTSTRUCT ps;
		BeginPaint(compHWND, &ps);

		EndPaint(compHWND, &ps);
	}

	void AddNowComponent(NOWComponent *component)
	{
		if (component)
			componentList->AddPointer(component);
	}

	NOWComponent* GetComponentAt(int xPos, int yPos)
	{
		for (int i = componentList->GetSize()-1; i>=0; i--) // search top to bottom
		{
			NOWComponent* component = componentList->GetPointer(i);
			if (component->IsVisible())
			{
				if (component->IsMouseWithinArea(xPos, yPos))
				{
					return component;
				}
			}
		}
		return 0;
	}

	void OnMouseLDown(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);

		if (component) // LDown on component!
		{
			component->OnMouseLDown(xPos, yPos); // send event!
		}
		else // LDown for client area drag
		{
			OnMLButtonDownForDrag(xPos, yPos);
		}
	}

	void OnMouseMove(int xPos, int yPos, WPARAM fwKeys)
	{
		if (windowDraging) // client area drag!
		{
			OnMouseMoveForDrag();
		}
		else // send event to appropriate component!
		{

			if (mouseFocusedComponent)
			{
				mouseFocusedComponent->OnMouseMove(xPos, yPos, fwKeys);
				return;
			}

			NOWComponent* component = GetComponentAt(xPos, yPos);

			if (component)// mouse is on component!
			{
				if (component == mouseOverComponent)// mouse is still on prev comp!
				{
					component->OnMouseMove(xPos, yPos, fwKeys);
				}
				else // mouse is on new component!
				{
					if (mouseOverComponent)// tell it to prev comp!
						mouseOverComponent->OnMouseLost(fwKeys);

					mouseOverComponent = component;// new mouse over comp!
					mouseOverComponent->OnMouseOver(fwKeys);
				}
			}
			else // mouse is not on component!
			{
				if (mouseOverComponent)// tell it to prev comp!
				{
					mouseOverComponent->OnMouseLost(fwKeys);
					mouseOverComponent = 0;
				}
			}


		}
	}

	void OnMouseLUp(int xPos, int yPos)
	{
		if (windowDraging) // client area drag!
		{
			OnMLButtonUpForDrag();
		}
		else // send event to appropriate component!
		{

			if (mouseFocusedComponent)
			{
				mouseFocusedComponent->OnMouseLUp(xPos, yPos);
				return;
			}

			NOWComponent* component = GetComponentAt(xPos, yPos);

			if (component)// mouse is on component!
			{
				component->OnMouseLUp(xPos, yPos);
			}

		}
	}

	void OnMouseRDown(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);
		if (component)
			component->OnMouseRDown(xPos, yPos);
	}

	void OnMouseRUp(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);
		if (component)
			component->OnMouseRUp(xPos, yPos);
	}

	void ConvertToTime(int totalSeconds, int *hours, int *mins, int *secs)
	{
		int _hours = totalSeconds / 3600;
		*hours = _hours;
		if (hours)
		{		
			int remainingSecs = totalSeconds - (_hours * 3600);
			int _mins = remainingSecs / 60;
			*mins = _mins;
			if (_mins)
			{
				remainingSecs = remainingSecs - (_mins * 60);
				*secs = remainingSecs;
			}
			else
			{
				*secs = remainingSecs;
			}
		}
		else
		{
			int _mins = totalSeconds / 60;
			*mins = _mins;
			if (_mins)
			{
				int remainingSecs = totalSeconds - (_mins * 60);
				*secs = remainingSecs;
			}
			else
			{
				*secs = totalSeconds;
			}
		}
	}

	void LoadFile(KString &filePath)
	{
		enableVis = false;

		// free old one
		if (hstream)
		{
			BASS_ChannelStop(hstream);
			BASS_StreamFree(hstream);
		}

		hstream = BASS_StreamCreateFile(FALSE, (const char*)filePath, 0, 0, 0);

		if (hstream) // read tags
		{
			char *title = (char*)TAGS_Read(hstream, "%TITL"); // do not free returned buffer
			titleLabel->SetText(KString((const char*)title, CP_UTF8));

			char *artist = (char*)TAGS_Read(hstream, "%ARTI"); // do not free returned buffer
			artistLabel->SetText(KString((const char*)artist, CP_UTF8));

			QWORD length = BASS_ChannelGetLength(hstream, BASS_POS_BYTE);
			double totalSecs = BASS_ChannelBytes2Seconds(hstream, length);

			int hours, mins, secs;
			ConvertToTime(totalSecs, &hours, &mins, &secs);
			totalTimeLabel->SetTime(hours, mins, secs);

		}

		enableVis = true;
	}

	void PlayFile()
	{
		if (hstream)
			BASS_ChannelPlay(hstream, FALSE);
	}

	void OnDropFile(WPARAM wParam, LPARAM lParam)
	{
		HDROP hdrop = (HDROP)wParam;
		UINT  uNumFiles;

		int charCount = DragQueryFileW(hdrop, 0, NULL, 0); // get the char count to hold file path

		wchar_t *szFile = (wchar_t*)malloc((charCount + 1) * sizeof(wchar_t));

		if (DragQueryFileW(hdrop, 0, szFile, charCount+1) > 0) // get the first file only
		{
			LoadFile(KString(szFile));
			PlayFile();
		}

		free(szFile);

		DragFinish(hdrop);
	}

	LRESULT WindowProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{
		switch (msg)
		{
		case WM_PAINT:
			this->OnPaint();
			break;
		case WM_ERASEBKGND:
			return 1;
		case WM_MOUSEMOVE:
			this->OnMouseMove(LOWORD(lParam), HIWORD(lParam), wParam);
			break;
		case WM_LBUTTONDOWN:
			this->OnMouseLDown(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_LBUTTONUP:
			this->OnMouseLUp(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_RBUTTONDOWN:
			this->OnMouseRDown(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_RBUTTONUP:
			this->OnMouseRUp(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_DROPFILES:
			this->OnDropFile(wParam, lParam);
			break;
		default:
			return KWindow::WindowProc(hwnd, msg, wParam, lParam);
		}

		return 0;
	}

	void OnMLButtonDownForDrag(int xPos, int yPos)
	{
		windowDraging = true;
		SetCapture(compHWND);

		cld_mouse_x = xPos;
		cld_mouse_y = yPos;
	}

	void OnMouseMoveForDrag()
	{
		POINT pos;
		GetCursorPos(&pos);
		SetPosition(pos.x - cld_mouse_x, pos.y - cld_mouse_y);
	}

	void OnMLButtonUpForDrag()
	{
		ReleaseCapture();
		windowDraging = false;
	}

	void releaseResources()
	{
		delete backBuffer;
		delete gradImage;
		delete backPanel;
		delete shapeImg;
		delete maskImg;
		delete ballImage;
		delete glassImg;
		delete linesImage;
		delete starsImg;
		delete visGridImg;

		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			delete particles[i];
		}

		for (int i = 0; i < componentList->GetSize(); i++)
			delete componentList->GetPointer(i);

		delete componentList;

		delete normalFont;
		delete headerFont;
		delete smallFont;
	}
};

class NOWPlayer : public KApplication
{
public:
	int Main(KString **argv, int argc)
	{

		MainWindow mainWindow;

		srand(time(NULL));

		mainWindow.init();
		mainWindow.SetVisible(true);

		DoMessagePump(false);

		mainWindow.releaseResources();

		return 0;
	}
};

START_RFC_APPLICATION(NOWPlayer)